library(ggplot2)
library(ggpubr)
library(openxlsx)

# Load the data
data <- read.xlsx("AA.xlsx")

# Convert 0 and 1 in 'group' to appropriate labels
data$group <- factor(data$group, levels = c(0, 1), labels = c("no AE", "AE"))

# Plot boxplot for the raw DHA data (without log transformation)
comparisons <- list(c("no AE", "AE"))

p <- ggplot(data = data, aes(x = group, y = AA)) +  # Use raw DHA data here
  geom_boxplot(aes(fill = NA), width = 0.6, size = 0.8, outlier.shape = NA, alpha = 0) +  # No fill color for boxes
  stat_boxplot(geom = "errorbar", width = 0.2, aes()) +
  geom_jitter(aes(fill = group), width = 0.2, shape = 21, size = 5) +
  scale_fill_manual(values = c("#B2DF8A", "#B2DF8A")) +
  scale_color_manual(values = c("#B2DF8A", "#B2DF8A")) +
  theme_bw() + 
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank()) +
  theme(axis.text = element_text(colour = 'black', size = 18)) +
  labs(x = '', y = 'EPA') +
  scale_y_continuous(labels = scales::scientific, breaks = seq(0, 0.00020, 0.00005), limits = c(0, 0.00020)) +
  theme(legend.position = "none")  # Remove the legend

# Statistical comparison
formatPvlaue <- function(p){ 
  if (p < 0.001) {
    return("p<0.001")
  } else {
    return(sprintf("p=%.3f", p))
  }
}

p <- p + geom_signif(comparisons = comparisons, 
                     map_signif_level = formatPvlaue, 
                     annotations = NULL,  
                     test = wilcox.test,  
                     y_position = c(0.00018), 
                     tip_length = 0,  
                     vjust = 1.6, 
                     color = "black", 
                     step_increase = 0.0, 
                     textsize = 5,  
                     size = 1)

# Customize the plot
p <- p + theme(text = element_text(size = 20, face = "bold"))

# Save the plot as PDF and JPG
ggsave(plot = p, filename = "output/AA.pdf", height = 7, width = 8)

# Print the plot
print(p)
